

#include "Bicubic.fxh"

//blend properties
//!!!!!!!!!!!!!!!!!!!!!
// Change here to match your blend region count.
// This should generally be equal your viewport*renderer count
#define nRegions 4
//!!!!!!!!!!!!!!!!!!!!!


// --------------------------------------------------------------------------------------------------
// PARAMETERS:
// --------------------------------------------------------------------------------------------------

//transforms
float4x4 tW: WORLD;        //the models world matrix
float4x4 tV: VIEW;         //view matrix as set via Renderer 
float4x4 tWVP: WORLDVIEWPROJECTION;

//texture
Texture2D Tex <string uiname="Texture";>;


float yBlends[nRegions-1];
float blendWidth=0.1;
float3 gammaRGB=1;
bool calibrate=0;
float lineWidth=0.01;
float3 blendVector=float3(0,1,0);


//texture projection properties
bool useProjection=false; // false=use vertex tex coords, true=use world coords
float4x4 tProjector;

//viewports
int ViewIndex: VIEWPORTINDEX;


SamplerState g_samLinear : IMMUTABLE
{
    Filter = MIN_MAG_MIP_LINEAR;
    AddressU = Clamp;
    AddressV = Clamp;
};


SamplerState g_samPoint : IMMUTABLE
{
    Filter = MIN_MAG_MIP_POINT;
    AddressU = Clamp;
    AddressV = Clamp;
};


//texture transformation marked with semantic TEXTUREMATRIX to achieve symmetric transformations

float4x4 tTex <bool uvspace=true; string uiname="Texture Transform";>;
//alpha
float Alpha <float uimin=0.0; float uimax=1.0;> = 1;

//the data structure: "vertexshader to pixelshader"
//used as output data with the VS function
//and as input data with the PS function
struct vs2ps
{
    float4 Pos  : SV_POSITION;
    float4 TexCd : TEXCOORD0;
    float4 PosW : TEXCOORD1;
};

// --------------------------------------------------------------------------------------------------
// VERTEXSHADERS
// --------------------------------------------------------------------------------------------------

vs2ps VS(
    float4 PosO  : POSITION,
    float4 TexCd : TEXCOORD0
    )
{
    //inititalize all fields of output struct with 0
    vs2ps Out = (vs2ps)0;

    //transform position
    Out.Pos = mul(PosO, tWVP);
    float4 PosW = mul(PosO, tW);

    //transform texturecoordinates
    if (useProjection)
    {
        Out.TexCd = mul(PosW, tProjector);
    }
    else
        Out.TexCd = mul(TexCd, tTex);
    
    Out.PosW = mul(PosO, tW);
    
    return Out;
}

// --------------------------------------------------------------------------------------------------
// PIXELSHADERS:
// --------------------------------------------------------------------------------------------------

float3 factor(float pos)
{
    float xTop = yBlends[ViewIndex-1];
    float xBase = yBlends[ViewIndex];


	float factortop = clamp((pos - (xTop-blendWidth/2)) /
					((xTop+blendWidth/2)-(xTop-blendWidth/2)), 0, 1);
	
    if (ViewIndex==0)
        factortop=1;


    float factorbottom = 1 - clamp((pos - (xBase-blendWidth/2)) /
					((xBase+blendWidth/2)-(xBase-blendWidth/2)), 0, 1);
	
    if (ViewIndex==nRegions-1)
        factorbottom=1;

	return factorbottom * factortop ;
	
   //return pow(factortop * factorbottom, gammaRGB);
}

float4 calPattern(float x)
{
    float top = yBlends[ViewIndex-1];
    float base = yBlends[ViewIndex];
    
    float4 col = 0;
    
    if (ViewIndex>0)
        col += 1-smoothstep(lineWidth/2,lineWidth,abs(top-x));
        
    if (ViewIndex<nRegions-1)
        col += 1-smoothstep(lineWidth/2,lineWidth,abs(base-x));

// checkerboard
//    col.a = ((PosW.xyz/0.2) - floor(PosW.xyz/0.2)>0.5).x;

 return col;
}

float4 PSGeneral(vs2ps In, float4 texCol)
{
  if (useProjection)
  {
  	In.TexCd /= In.TexCd.w;
  	In.TexCd.xy+=1;
  	In.TexCd.xy /=2;
  	In.TexCd.y=1-In.TexCd.y;
  }

  float4 col = texCol;
  
  col.a *= Alpha;
  
  float x = dot(In.PosW.xyz,blendVector);
	
  col.rgb *= factor(x);

  if (calibrate)
     col += calPattern(x);

  return col;
}

 //PS Bicubic
float4 PSbic(vs2ps In) : SV_Target
{
	return PSGeneral(In, tex2Dbicubic(Tex,g_samLinear,(float2)In.TexCd));
}

float4 PSlin(vs2ps In) : SV_Target
{
	return PSGeneral(In, tex2Dbicubic(Tex,g_samLinear, (float2)In.TexCd));
}
  
// PS Nearest Neighbour
float4 PSnn(vs2ps In) : SV_Target
{
	return PSGeneral(In, Tex.Sample(g_samPoint,(float2) In.TexCd));
}



technique10 Bicubic
{
	pass P0
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );
		SetPixelShader( CompileShader( ps_4_0, PSbic() ) );
	}
}

technique10 Bilinear
{
	pass P0
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );
		SetPixelShader( CompileShader( ps_4_0, PSlin() ) );
	}
}

technique10 NearestNeighbour
{
	pass P0
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );
		SetPixelShader( CompileShader( ps_4_0, PSnn() ) );
	}
}

